// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/metrics/desktop_platform_features_metrics_provider.h"

#include <vector>

#include "base/feature_list.h"
#include "base/metrics/histogram_macros.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/profiles/profile_manager.h"
#include "chrome/browser/ui/browser.h"
#include "chrome/browser/ui/browser_list.h"
#include "chrome/browser/ui/read_later/reading_list_model_factory.h"
#include "components/reading_list/core/reading_list_model.h"
#include "components/reading_list/features/reading_list_switches.h"
#include "ui/native_theme/native_theme.h"

namespace {

// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
enum class DarkModeStatus {
  kUnavailable = 0,
  kLight = 1,
  kDark = 2,
  kMaxValue = kDark,
};

bool AnyBrowserWindowHasName() {
  for (auto* browser : *BrowserList::GetInstance()) {
    if (!browser->user_title().empty())
      return true;
  }
  return false;
}

}  // namespace

DesktopPlatformFeaturesMetricsProvider::
    DesktopPlatformFeaturesMetricsProvider() {}
DesktopPlatformFeaturesMetricsProvider::
    ~DesktopPlatformFeaturesMetricsProvider() {}

void DesktopPlatformFeaturesMetricsProvider::ProvideCurrentSessionData(
    metrics::ChromeUserMetricsExtension* uma_proto) {
  ui::NativeTheme* theme = ui::NativeTheme::GetInstanceForNativeUi();
  DarkModeStatus status = DarkModeStatus::kUnavailable;
  if (ui::NativeTheme::SystemDarkModeSupported()) {
    status = theme->ShouldUseDarkColors() ? DarkModeStatus::kDark
                                          : DarkModeStatus::kLight;
  }
  UMA_HISTOGRAM_ENUMERATION("Browser.DarkModeStatus", status);

  // Record how many items are in the reading list.
  if (base::FeatureList::IsEnabled(reading_list::switches::kReadLater)) {
    std::vector<Profile*> profiles =
        g_browser_process->profile_manager()->GetLoadedProfiles();
    for (Profile* profile : profiles) {
      ReadingListModel* model =
          ReadingListModelFactory::GetForBrowserContext(profile);
      if (model && model->loaded()) {
        UMA_HISTOGRAM_COUNTS_1000("ReadingList.Unread.Count.OnUMAUpload",
                                  model->unread_size());
        UMA_HISTOGRAM_COUNTS_1000("ReadingList.Read.Count.OnUMAUpload",
                                  model->size() - model->unread_size());
      }
    }
  }

  UMA_HISTOGRAM_BOOLEAN("Browser.AnyWindowHasName", AnyBrowserWindowHasName());
}
