// Copyright (c) 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/components/audio/audio_devices_pref_handler_stub.h"

#include "ash/components/audio/audio_device.h"
#include "base/containers/contains.h"

namespace ash {

AudioDevicesPrefHandlerStub::AudioDevicesPrefHandlerStub() = default;

AudioDevicesPrefHandlerStub::~AudioDevicesPrefHandlerStub() = default;

double AudioDevicesPrefHandlerStub::GetOutputVolumeValue(
    const AudioDevice* device) {
  if (!device || !base::Contains(audio_device_volume_gain_map_,
                                 device->stable_device_id)) {
    return kDefaultOutputVolumePercent;
  }
  return audio_device_volume_gain_map_[device->stable_device_id];
}

double AudioDevicesPrefHandlerStub::GetInputGainValue(
    const AudioDevice* device) {
  // TODO(rkc): The default value for gain is wrong. http://crbug.com/442489
  if (!device || !base::Contains(audio_device_volume_gain_map_,
                                 device->stable_device_id)) {
    if (device->is_input) {
      return 50.0;
    }
    return 75.0;
  }
  return audio_device_volume_gain_map_[device->stable_device_id];
}

void AudioDevicesPrefHandlerStub::SetVolumeGainValue(const AudioDevice& device,
                                                     double value) {
  audio_device_volume_gain_map_[device.stable_device_id] = value;
}

bool AudioDevicesPrefHandlerStub::GetMuteValue(
    const AudioDevice& device) {
  return audio_device_mute_map_[device.stable_device_id];
}

void AudioDevicesPrefHandlerStub::SetMuteValue(const AudioDevice& device,
                                               bool mute_on) {
  audio_device_mute_map_[device.stable_device_id] = mute_on;
}

void AudioDevicesPrefHandlerStub::SetDeviceActive(const AudioDevice& device,
                                                  bool active,
                                                  bool activate_by_user) {
  DeviceState state;
  state.active = active;
  state.activate_by_user = activate_by_user;
  audio_device_state_map_[device.stable_device_id] = state;
}

bool AudioDevicesPrefHandlerStub::GetDeviceActive(const AudioDevice& device,
                                                  bool* active,
                                                  bool* activate_by_user) {
  if (audio_device_state_map_.find(device.stable_device_id) ==
      audio_device_state_map_.end()) {
    return false;
  }
  *active = audio_device_state_map_[device.stable_device_id].active;
  *activate_by_user =
      audio_device_state_map_[device.stable_device_id].activate_by_user;
  return true;
}
bool AudioDevicesPrefHandlerStub::GetNoiseCancellationState() {
  return noise_cancellation_state_;
}

void AudioDevicesPrefHandlerStub::SetNoiseCancellationState(
    bool noise_cancellation_state) {
  noise_cancellation_state_ = noise_cancellation_state;
}

bool AudioDevicesPrefHandlerStub::GetAudioOutputAllowedValue() {
  return true;
}

void AudioDevicesPrefHandlerStub::AddAudioPrefObserver(
    AudioPrefObserver* observer) {
}

void AudioDevicesPrefHandlerStub::RemoveAudioPrefObserver(
    AudioPrefObserver* observer) {
}

}  // namespace ash
