"""Test the `tinydns` and `dnsq` programs."""

import dataclasses
import enum
import os
import pathlib
import subprocess
import sys
import time

from typing import Callable, Dict, List, Tuple, Union

from . import defs
from . import test_get


class ProcessFDType(str, enum.Enum):
    """File descriptor types, enough for this very limited test."""

    IPV4 = "IPv4"


@dataclasses.dataclass(frozen=True)
class ProcessFDInfo:
    """Information about a single file descriptor in a process."""

    fileno: int
    access: str
    lock: str
    ftype: ProcessFDType


@dataclasses.dataclass(frozen=True)
class ProcessSocketFDInfo(ProcessFDInfo):
    """Information about a single socket in a process."""

    proto: str
    address: str
    port: int


@dataclasses.dataclass(frozen=True)
class ProcessInfo:
    """Information about a process and (some of) its file descriptors."""

    pid: int
    pgid: int
    tid: int
    comm: str

    uid: int
    username: str

    files: List[ProcessFDInfo]


class ParseProcessInfo:
    """Parse the output of `lsof -F` for our very limited testing needs."""

    cfg: defs.Config
    cproc: Dict[str, str]
    cfile: Dict[str, str]
    cfiles: List[ProcessFDInfo]
    result: Dict[int, ProcessInfo]

    def __init__(self, cfg: defs.Config) -> None:
        """Initialize an empty object."""
        self.cfg = cfg
        self.cproc = {}
        self.cfile = {}
        self.cfiles = []
        self.result = {}

    def finish_file_ipv4(self) -> ProcessFDInfo:
        """Store the information about a socket."""
        fields = self.cfile["addrport"].split(":", 1)
        return ProcessSocketFDInfo(
            fileno=int(self.cfile["fileno"]),
            access=self.cfile["access"],
            lock=self.cfile["lock"],
            ftype=ProcessFDType.IPV4,
            proto=self.cfile["proto"],
            address=fields[0],
            port=int(fields[1]),
        )

    def finish_file(self) -> None:
        """Store the information about the current file descriptor."""
        if self.cfile["type"] == "IPv4":
            fdata = self.finish_file_ipv4()
        else:
            sys.exit(f"Internal error: unhandled file type: {self.cfile!r}")

        self.cfile.clear()

        self.cfiles.append(fdata)

    def finish_process(self) -> None:
        """Store the information about the current process."""
        if self.cfile:
            self.finish_file()

        pdata = ProcessInfo(
            pid=int(self.cproc["pid"]),
            pgid=int(self.cproc["pgid"]),
            tid=int(self.cproc["tid"]),
            comm=self.cproc["comm"],
            uid=int(self.cproc["uid"]),
            username=self.cproc["username"],
            files=list(self.cfiles),
        )

        self.cfiles.clear()
        self.cproc.clear()

        if pdata.pid in self.result:
            sys.exit(
                f"lsof output a duplicate process set for {pdata.pid}: "
                f"first {self.result[pdata.pid]!r} now {pdata!r}"
            )
        self.result[pdata.pid] = pdata

    def parse(self, lines: List[str]) -> Dict[int, ProcessInfo]:
        """Parse some lsof output lines."""

        def h_process(_first: str, rest: str) -> None:
            """A new process set."""
            if self.cproc:
                self.finish_process()

            self.cproc["pid"] = rest

        def h_file(_first: str, rest: str) -> None:
            """A new file descriptor set."""
            if self.cfile:
                self.finish_file()

            self.cfile["fileno"] = rest

        handlers: Dict[
            str, Union[Callable[[str, str], None], Tuple[Dict[str, str], str]]
        ] = {
            "p": h_process,
            "g": (self.cproc, "pgid"),
            "R": (self.cproc, "tid"),
            "c": (self.cproc, "comm"),
            "u": (self.cproc, "uid"),
            "L": (self.cproc, "username"),
            "f": h_file,
            "a": (self.cfile, "access"),
            "l": (self.cfile, "lock"),
            "t": (self.cfile, "type"),
            "P": (self.cfile, "proto"),
            "n": (self.cfile, "addrport"),
        }

        for line in lines:
            if not line:
                sys.exit("lsof output an empty line")
            first, rest = line[0], line[1:]
            handler = handlers.get(first)
            if not handler:
                continue

            if isinstance(handler, tuple):
                handler[0][handler[1]] = rest
            else:
                handler(first, rest)

        if self.cproc:
            self.finish_process()

        return self.result


def check_single_udp_socket(
    processes: Dict[int, ProcessInfo], pid: int, ipaddr: str
) -> bool:
    """Make sure tinydns is only listening on a single UDP socket."""
    pdata = processes.get(pid)
    if pdata is None or len(pdata.files) != 1:
        return False

    data = pdata.files[0]
    if not isinstance(data, ProcessSocketFDInfo):
        return False

    return data.proto == "UDP" and data.address == ipaddr and data.port == 53


def test_tinydns_run(cfg: defs.Config, tempd: pathlib.Path) -> None:
    """Test the `tinydns` and `dnsq` programs."""
    print("\n==== test_tinydns_run")
    if os.geteuid() != 0:
        print("- not running as root, skipped")
        return

    dnsq = cfg.bindir / "dnsq"
    if not dnsq.is_file() or not os.access(dnsq, os.X_OK):
        sys.exit(f"Not an executable file: {dnsq}")

    ipaddr = defs.RECORDS[0].address
    svcdir = tempd / defs.SVCDIR
    if not svcdir.is_dir():
        sys.exit(f"Expected {svcdir} to be a directory")
    os.chdir(svcdir)

    proc = None
    try:
        proc = subprocess.Popen(
            ["./run"],
            shell=False,
            env=cfg.subenv,
            bufsize=0,
            encoding=defs.MINENC,
            stdout=subprocess.PIPE,
        )
        print(f"- spawned process {proc.pid}")
        assert proc.stdout is not None
        print("- waiting for the 'starting tinydns' line")
        line = proc.stdout.readline()
        print(f"- got line {line!r}")
        if line != "starting tinydns\n":
            sys.exit(f"Unexpected first line from tinydns: {line!r}")

        lines = cfg.check_output(
            ["lsof", "-a", "-n", "-P", "-p", str(proc.pid), "-F", "-i4udp:53"]
        ).splitlines()
        cfg.diag(f"lsof output: {lines!r}")
        processes = ParseProcessInfo(cfg).parse(lines)
        if not check_single_udp_socket(processes, proc.pid, ipaddr):
            sys.exit(
                f"tinydns should listen on a single UDP socket: {processes!r}"
            )
        print(f"- tinydns is listening at {ipaddr}:53")

        for rec in defs.RECORDS:
            rdef = defs.TYPES[rec.rtype]
            print(f"- querying {ipaddr} for {rdef.query} {rec.name}")
            test_get.test_tinydns_query(
                cfg,
                [
                    "timelimit",
                    "-p",
                    "-t1",
                    "-T1",
                    "--",
                    dnsq,
                    rdef.query,
                    rec.name,
                    ipaddr,
                ],
                rec,
            )

            print("  - there should be a single line of output from tinydns")
            line = proc.stdout.readline()
            cfg.diag(f"line: {line!r}")
    finally:
        if proc is not None:
            print("- we spawned a process, checking if it has exited")
            res = proc.poll()
            if res is None:
                print(f"Terminating process {proc.pid}")
                proc.terminate()
                time.sleep(0.5)
                res = proc.poll()
                if res is None:
                    print(f"Killing process {proc.pid}")
                    proc.kill()
            res = proc.wait()
            print(f"Process {proc.pid}: exit code {res}")
